-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
source(thModPath .. "scripts/utilities/THUtils.lua")
source(thModPath .. "scripts/utilities/THDebugUtil.lua")
source(thModPath .. "scripts/managers/THModManager.lua")
THDefaultTypes = {}
local THDefaultTypes_mt = THUtils.createClass(THDefaultTypes)
local debugFlagId = THDebugUtil.createFlagId("THDefaultTypes")
THDefaultTypes.debugFlagId = debugFlagId
THDebugUtil.setIsEnabled(debugFlagId, true)
THDefaultTypes.DATA_KEY = "thDefaultTypes"
THDefaultTypes.XML_KEY = THDefaultTypes.DATA_KEY
function THDefaultTypes.new(customMt)
    customMt = customMt or THDefaultTypes_mt
    if THUtils.argIsValid(type(customMt) == "table", "customMt", customMt) then
        local self = setmetatable({}, customMt)
        self.isServer = g_server ~= nil
        self.isClient = g_client ~= nil
        self.modName = thModName
        self.modPath = thModPath
        self.dataKey = THDefaultTypes.DATA_KEY
        self.xmlKey = THDefaultTypes.XML_KEY
        self.taskManager = g_asyncTaskManager
        self.fruitTypeManager = g_fruitTypeManager
        self.fillTypeManager = g_fillTypeManager
        self.thModManager = g_thModManager
        self.defaultTypes = {}
        return self
    end
end
function THDefaultTypes.loadDefaultTypeData(self, typeName, missionInfo, xmlFile)
    if THUtils.argIsValid(type(typeName) == THValueType.STRING, "typeName", typeName)
        and THUtils.argIsValid(type(missionInfo) == THValueType.TABLE, "missionInfo", missionInfo)
    then
        local defaultTypeData = self.defaultTypes[typeName]
        local success = true
        if defaultTypeData == nil then
            local mapDesc = missionInfo.map
            local xmlRootKey = string.format("map.%s", self.xmlKey)
            if mapDesc == nil then
                THUtils.errorMsg(true, "Could not find map information")
                success = false
            else
                local mapXMLFile, _, mapModName, mapModPath, isMapSettings = THUtils.loadMapXMLFile(mapDesc, "THDefaultTypesXML", nil, false, true)
                if not isMapSettings and xmlFile ~= nil then
                    if mapXMLFile ~= nil then
                        THUtils.deleteXMLFile(mapXMLFile)
                    end
                    mapXMLFile = xmlFile
                end
                if mapXMLFile ~= nil
                    and mapModName ~= nil and mapModName ~= ""
                    and mapModPath ~= nil and mapModPath ~= ""
                then
                    defaultTypeData = {
                        typeName = typeName,
                        mapXMLFile = xmlFile,
                        mapModName = mapModName,
                        mapModPath = mapModPath,
                        isMapLoading = false,
                        isDefaultLoading = false,
                        isOldVersion = false,
                        blackListItems = {}
                    }
                    defaultTypeData.getIsBlackListItem = function(pSelf, pItemId, pModName)
                        return self:getIsBlackListItem(pSelf, pItemId, pModName)
                    end
                    defaultTypeData.addBlackListItem = function(pSelf, pItemId, pModName)
                        return self:addBlackListItem(pSelf, pItemId, pModName)
                    end
                    if self:loadBlackListItems(defaultTypeData, mapXMLFile, mapModName, mapModPath) then
                        local dataXMLKey = xmlRootKey .. "." .. typeName
                        local dataXMLFilename = THUtils.getXMLValue(mapXMLFile, XMLValueType.STRING, dataXMLKey, "#filename")
                        if dataXMLFilename == nil then
                            local oldDataXMLKey = "map." .. typeName
                            local isDefaultType = THUtils.getXMLValue(mapXMLFile, XMLValueType.BOOL, oldDataXMLKey, "#" .. self.dataKey, false)
                            if isDefaultType then
                                dataXMLKey = oldDataXMLKey
                                dataXMLFilename = THUtils.getXMLValue(mapXMLFile, XMLValueType.STRING, dataXMLKey, "#filename")
                                if dataXMLFilename ~= nil then
                                    defaultTypeData.isOldVersion = true
                                else
                                    THUtils.xmlErrorMsg(dataXMLKey, nil, "Filename required for backwards compatible functionality")
                                    THUtils.errorMsg(nil, "- Please upgrade your configuration to the new version")
                                end
                            end
                        end
                        if dataXMLFilename ~= nil then
                            dataXMLFilename = THUtils.getFilename(dataXMLFilename, mapModPath)
                            if THUtils.getFileExists(dataXMLFilename) then
                                local dataXMLFile = THUtils.loadXMLFile("THDefaultTypesDataXML", dataXMLFilename, nil, false)
                                if dataXMLFile ~= nil then
                                    local dataXMLRootKey = "map." .. typeName
                                    if typeName == "storeItems" then
                                        dataXMLRootKey = typeName
                                    end
                                    defaultTypeData.xmlFile = dataXMLFile
                                    defaultTypeData.xmlFilename = dataXMLFilename
                                    defaultTypeData.xmlRootKey = dataXMLRootKey
                                    if defaultTypeData.isOldVersion and mapXMLFile == xmlFile then
                                        THUtils.removeXMLProperty(xmlFile, dataXMLKey .. "#filename")
                                    end
                                end
                            else
                                THUtils.xmlErrorMsg(dataXMLKey, nil, THMessage.FILE_NOT_FOUND, dataXMLFilename)
                            end
                        end
                    else
                        success = false
                    end
                    if mapXMLFile ~= xmlFile then
                        THUtils.deleteXMLFile(mapXMLFile)
                    end
                end
            end
            if success and defaultTypeData ~= nil then
                self.defaultTypes[defaultTypeData.typeName] = defaultTypeData
            end
        end
        if success and defaultTypeData ~= nil then
            if THDebugUtil.getIsEnabled(debugFlagId) then
                THUtils.displayMsg("Loading type data: %s", defaultTypeData.typeName)
                THDebugUtil.printTable(defaultTypeData, 2)
            end
            return defaultTypeData
        end
    end
end
function THDefaultTypes.getIsBlackListItem(self, defaultTypeData, itemId, modName)
    if THUtils.argIsValid(type(defaultTypeData) == THValueType.TABLE and defaultTypeData.typeName ~= nil, "defaultTypeData", defaultTypeData)
        and THUtils.argIsValid(modName == nil or type(modName) == "string", "modName", modName)
    then
        if itemId ~= nil then
            local baseDirectory = nil
            if modName ~= nil then
                local modData = self.thModManager:getLoadedMod(modName)
                if modData ~= nil then
                    baseDirectory = modData.path
                end
            end
            if defaultTypeData.typeName == "storeItems" then
                if THUtils.argIsValid(type(itemId) == THValueType.STRING, "itemId", itemId) then
                    local absXMLFilename = THUtils.getFilename(itemId, baseDirectory)
                    return defaultTypeData.blackListItems[absXMLFilename] == true
                end
            else
                if THUtils.argIsValid(type(itemId) == THValueType.STRING, "itemId", itemId) then
                    return defaultTypeData.blackListItems[itemId:upper()] == true
                end
            end
        end
    end
    return false
end
function THDefaultTypes.addBlackListItem(self, defaultTypeData, itemId, modName)
    if THUtils.argIsValid(type(defaultTypeData) == THValueType.TABLE and defaultTypeData.typeName ~= nil, "defaultTypeData", defaultTypeData)
        and THUtils.argIsValid(modName == nil or type(modName) == "string", "modName", modName)
    then
        if itemId ~= nil then
            local baseDirectory = nil
            local success = false
            if modName ~= nil then
                local modData = self.thModManager:getLoadedMod(modName)
                if modData ~= nil then
                    baseDirectory = modData.path
                end
            end
            if defaultTypeData.typeName == "storeItems" then
                if THUtils.argIsValid(type(itemId) == THValueType.STRING, "itemId", itemId) then
                    local absXMLFilename = THUtils.getFilename(itemId, baseDirectory)
                    defaultTypeData.blackListItems[absXMLFilename] = true
                    success = true
                end
            else
                if THUtils.argIsValid(type(itemId) == THValueType.STRING, "itemId", itemId) then
                    defaultTypeData.blackListItems[itemId:upper()] = true
                    success = true
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.loadBlackListItems(self, defaultTypeData, mapXMLFile, customEnv, baseDirectory)
    if THUtils.argIsValid(type(defaultTypeData) == THValueType.TABLE and defaultTypeData.typeName ~= nil, "defaultTypeData", defaultTypeData)
        and THUtils.argIsValid(THUtils.getIsType(mapXMLFile, "table|integer"), "mapXMLFile", mapXMLFile)
        and THUtils.argIsValid(customEnv == nil or type(customEnv) == THValueType.STRING, "customEnv", customEnv)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        local success = true
        if self.blackListXMLFile == nil then
            local blackListKey = string.format("map.%s.blackList", self.xmlKey)
            local blackListFilename = THUtils.getXMLValue(mapXMLFile, XMLValueType.STRING, blackListKey, "#filename")
            if blackListFilename ~= nil and blackListFilename ~= "" then
                local blackListXMLFile = THUtils.loadXMLFile("THBlackListXML", blackListFilename, baseDirectory, false)
                if blackListXMLFile ~= nil then
                    self.blackListXMLFile = blackListXMLFile
                    if THDebugUtil.getIsEnabled(debugFlagId) then
                        THUtils.displayMsg("Updating blacklist xml file: %s", blackListXMLFile)
                        THDebugUtil.printTable(self)
                    end
                else
                    THUtils.xmlErrorMsg(blackListKey, nil, THMessage.FILE_NOT_FOUND, blackListFilename)
                    success = false
                end
            end
        end
        if success and self.blackListXMLFile ~= nil then
            local defaultTypeName = defaultTypeData.typeName
            local defaultItemsKey = string.format("%s.blackList.%s", self.xmlKey, defaultTypeName)
            local itemXMLIndex = 0
            while true do
                local itemKey = nil
                if defaultTypeName == "fillTypes" then
                    itemKey = string.format("%s.fillType(%d)", defaultItemsKey, itemXMLIndex)
                elseif defaultTypeName == "fruitTypes" then
                    itemKey = string.format("%s.fruitType(%d)", defaultItemsKey, itemXMLIndex)
                elseif defaultTypeName == "densityMapHeightTypes" then
                    itemKey = string.format("%s.densityMapHeightType(%d)", defaultItemsKey, itemXMLIndex)
                elseif defaultTypeName == "storeItems" then
                    itemKey = string.format("%s.storeItem(%d)", defaultItemsKey, itemXMLIndex)
                end
                if itemKey == nil or not THUtils.hasXMLProperty(self.blackListXMLFile, itemKey) then
                    break
                end
                local itemNameKey = "#name"
                local itemModName = nil
                if defaultTypeName == "storeItems" then
                    itemNameKey = "#xmlFilename"
                    itemModName = THUtils.getXMLValue(self.blackListXMLFile, XMLValueType.STRING, itemKey, "#modName")
                elseif defaultTypeName == "densityMapHeightTypes" then
                    itemNameKey = "#fillTypeName"
                end
                local itemName = THUtils.getXMLValue(self.blackListXMLFile, XMLValueType.STRING, itemKey, itemNameKey)
                if itemName == nil or itemName == "" then
                    THUtils.xmlErrorMsg(itemKey, nil, THMessage.INVALID_VALUE, itemNameKey, itemName)
                else
                    defaultTypeData:addBlackListItem(itemName, itemModName)
                end
                itemXMLIndex = itemXMLIndex + 1
            end
        end
        return success
    end
    return false
end
function THDefaultTypes.getFillType(self, fillTypeId, manager)
    manager = manager or self.fillTypeManager
    local idType = type(fillTypeId)
    if idType == THValueType.STRING then
        return manager:getFillTypeByName(fillTypeId)
    elseif idType == THValueType.NUMBER then
        return manager:getFillTypeByIndex(fillTypeId)
    elseif fillTypeId ~= nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fillTypeId", fillTypeId)
    end
end
function THDefaultTypes.updateFillTypeXMLEntries(self, xmlFile)
    local defaultTypeData = self.defaultTypes.fillTypes
    if xmlFile ~= nil and defaultTypeData ~= nil
        and next(defaultTypeData.blackListItems) ~= nil
    then
        local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            while true do
                local entryKey = string.format("%s.fillTypeCategories.fillTypeCategory(%d)", xmlRootKey, xmlIndex)
                local isEntryFound = false
                if THUtils.hasXMLProperty(xmlFile, entryKey) then
                    isEntryFound = true
                    local fillTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, entryKey)
                    if fillTypeNames ~= nil and fillTypeNames ~= "" then
                        local namesList = THUtils.splitString(fillTypeNames, " ", true)
                        if namesList ~= nil then
                            fillTypeNames = nil
                            local isEntryKeyShown = false
                            for itemName in pairs(namesList) do
                                if not defaultTypeData:getIsBlackListItem(itemName) then
                                    if fillTypeNames == nil then
                                        fillTypeNames = itemName
                                    else
                                        fillTypeNames = fillTypeNames .. " " .. itemName
                                    end
                                elseif THDebugUtil.getIsEnabled(debugFlagId) then
                                    if not isEntryKeyShown then
                                        THUtils.displayMsg(entryKey)
                                        isEntryKeyShown = true
                                    end
                                    THUtils.displayMsg("- Removing fill type: %s", itemName)
                                end
                            end
                            if fillTypeNames == nil or fillTypeNames == "" then
                                fillTypeNames = " "
                            end
                            THUtils.setXMLValue(xmlFile, XMLValueType.STRING, entryKey, nil, fillTypeNames)
                        end
                    end
                end
                entryKey = string.format("%s.fillTypeConverters.fillTypeConverter(%d)", xmlRootKey, xmlIndex)
                if THUtils.hasXMLProperty(xmlFile, entryKey) then
                    isEntryFound = true
                    local converterXMLIndex = 0
                    while true do
                        local converterKey = string.format("%s.converter(%d)", entryKey, converterXMLIndex)
                        local isConverterRemoved = false
                        if not THUtils.hasXMLProperty(xmlFile, converterKey) then
                            break
                        end
                        local fromFillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, converterKey, "#from")
                        local toFillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, converterKey, "#to")
                        if (fromFillTypeName ~= nil and defaultTypeData:getIsBlackListItem(fromFillTypeName))
                            or (toFillTypeName ~= nil and defaultTypeData:getIsBlackListItem(toFillTypeName))
                        then
                            isConverterRemoved = true
                            isConverterRemoved = true
                        end
                        if isConverterRemoved then
                            THUtils.removeXMLProperty(xmlFile, converterKey)
                            if THDebugUtil.getIsEnabled(debugFlagId) then
                                THUtils.displayMsg(converterKey)
                                THUtils.displayMsg("- Entry removed: %s >> %s", fromFillTypeName, toFillTypeName)
                            end
                        else
                            converterXMLIndex = converterXMLIndex + 1
                        end
                    end
                end
                entryKey = string.format("%s.fillTypeSounds.fillTypeSound(%d)", xmlRootKey, xmlIndex)
                if THUtils.hasXMLProperty(xmlFile, entryKey) then
                    isEntryFound = true
                    local fillTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, entryKey, "#fillTypes")
                    if fillTypeNames ~= nil and fillTypeNames ~= "" then
                        local namesList = THUtils.splitString(fillTypeNames, " ", true)
                        if namesList ~= nil then
                            fillTypeNames = nil
                            local isEntryKeyShown = false
                            for itemName in pairs(namesList) do
                                if not defaultTypeData:getIsBlackListItem(itemName) then
                                    if fillTypeNames == nil then
                                        fillTypeNames = itemName
                                    else
                                        fillTypeNames = fillTypeNames .. " " .. itemName
                                    end
                                elseif THDebugUtil.getIsEnabled(debugFlagId) then
                                    if not isEntryKeyShown then
                                        THUtils.displayMsg(entryKey)
                                        isEntryKeyShown = true
                                    end
                                    THUtils.displayMsg("- Removing fill type: %s", itemName)
                                end
                            end
                            if fillTypeNames == nil or fillTypeNames == "" then
                                fillTypeNames = " "
                            end
                            THUtils.setXMLValue(xmlFile, XMLValueType.STRING, entryKey, "#fillTypes", fillTypeNames)
                        end
                    end
                end
                if not isEntryFound then
                    break
                end
                xmlIndex = xmlIndex + 1
            end
            return true
        end
    end
    return false
end
function THDefaultTypes.getFruitType(self, fruitTypeId, manager)
    manager = manager or self.fruitTypeManager
    local idType = type(fruitTypeId)
    if idType == THValueType.STRING then
        return manager:getFruitTypeByName(fruitTypeId)
    elseif idType == THValueType.NUMBER then
        return manager:getFruitTypeByIndex(fruitTypeId)
    elseif fruitTypeId ~= nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitTypeId", fruitTypeId)
    end
end
function THDefaultTypes.updateFruitTypeXMLEntries(self, xmlFile)
    local defaultTypeData = self.defaultTypes.fruitTypes
    local defaultFillTypeData = self.defaultTypes.fillTypes
    if xmlFile ~= nil and defaultTypeData ~= nil
        and next(defaultTypeData.blackListItems) ~= nil
    then
        local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            while true do
                local entryKey = string.format("%s.fruitTypeCategories.fruitTypeCategory(%d)", xmlRootKey, xmlIndex)
                local isEntryFound = false
                if THUtils.hasXMLProperty(xmlFile, entryKey) then
                    isEntryFound = true
                    local fruitTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, entryKey)
                    if fruitTypeNames ~= nil and fruitTypeNames ~= "" then
                        local namesList = THUtils.splitString(fruitTypeNames, " ", true)
                        if namesList ~= nil then
                            fruitTypeNames = nil
                            local isEntryKeyShown = false
                            for itemName in pairs(namesList) do
                                if not defaultTypeData:getIsBlackListItem(itemName) then
                                    if fruitTypeNames == nil then
                                        fruitTypeNames = itemName
                                    else
                                        fruitTypeNames = fruitTypeNames .. " " .. itemName
                                    end
                                elseif THDebugUtil.getIsEnabled(debugFlagId) then
                                    if not isEntryKeyShown then
                                        THUtils.displayMsg(entryKey)
                                        isEntryKeyShown = true
                                    end
                                    THUtils.displayMsg("- Removing fruit type: %s", itemName)
                                end
                            end
                            if fruitTypeNames == nil or fruitTypeNames == "" then
                                fruitTypeNames = " "
                            end
                            THUtils.setXMLValue(xmlFile, XMLValueType.STRING, entryKey, nil, fruitTypeNames)
                        end
                    end
                end
                entryKey = string.format("%s.fruitTypeConverters.fruitTypeConverter(%d)", xmlRootKey, xmlIndex)
                if THUtils.hasXMLProperty(xmlFile, entryKey) then
                    isEntryFound = true
                    local converterXMLIndex = 0
                    while true do
                        local converterKey = string.format("%s.converter(%d)", entryKey, converterXMLIndex)
                        local isConverterRemoved = false
                        if not THUtils.hasXMLProperty(xmlFile, converterKey) then
                            break
                        end
                        local fromFruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, converterKey, "#from")
                        local toFillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, converterKey, "#to")
                        if fromFruitTypeName ~= nil and defaultTypeData:getIsBlackListItem(fromFruitTypeName) then
                            isConverterRemoved = true
                        else
                            if defaultFillTypeData ~= nil
                                and toFillTypeName ~= nil and defaultFillTypeData:getIsBlackListItem(toFillTypeName)
                            then
                                isConverterRemoved = true
                            end
                        end
                        if isConverterRemoved then
                            THUtils.removeXMLProperty(xmlFile, converterKey)
                            if THDebugUtil.getIsEnabled(debugFlagId) then
                                THUtils.displayMsg(converterKey)
                                THUtils.displayMsg("- Entry removed: %s >> %s", fromFruitTypeName, toFillTypeName)
                            end
                        else
                            converterXMLIndex = converterXMLIndex + 1
                        end
                    end
                end
                if not isEntryFound then
                    break
                end
                xmlIndex = xmlIndex + 1
            end
            return true
        end
    end
    return false
end
function THDefaultTypes.inj_loadFruitTypeMapData(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, ...)
    if xmlFile ~= nil and missionInfo ~= nil then
        THUtils.call(function()
            if self.defaultTypes.fruitTypes == nil then
                self:loadDefaultTypeData("fruitTypes", missionInfo, xmlFile)
            end
            local defaultTypeData = self.defaultTypes.fruitTypes
            if defaultTypeData ~= nil and defaultTypeData.mapXMLFile == xmlFile then
                defaultTypeData.isMapLoading = true
            end
        end)
    end
    local function appendFunc(...)
        local defaultTypeData = self.defaultTypes.fruitTypes
        if defaultTypeData ~= nil then
            defaultTypeData.isMapLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, ...))
end
function THDefaultTypes.inj_loadDefaultFruitTypes(self, superFunc, manager, ...)
    local defaultTypeData = self.defaultTypes.fruitTypes
    if defaultTypeData ~= nil and defaultTypeData.xmlFile ~= nil then
        defaultTypeData.isDefaultLoading = true
    end
    local function appendFunc(...)
        if defaultTypeData ~= nil then
            defaultTypeData.isDefaultLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, ...))
end
function THDefaultTypes.inj_loadFruitTypes(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, isBaseType, ...)
    local defaultTypeData = self.defaultTypes.fruitTypes
    local isUserDefaults = false
    local oldXMLFile = xmlFile
    local oldBaseDirectory = baseDirectory
    local funcArgs = THUtils.pack(...)
    if defaultTypeData ~= nil then
        THUtils.call(function()
            if isBaseType == true
                and defaultTypeData.isDefaultLoading
                and defaultTypeData.xmlFile ~= nil
            then
                xmlFile = defaultTypeData.xmlFile
                baseDirectory = defaultTypeData.mapModPath
                THUtils.displayMsg("Loading default fruit types from: %s", defaultTypeData.xmlFilename)
                isUserDefaults = true
            end
            if xmlFile ~= nil and next(defaultTypeData.blackListItems) ~= nil then
                self:updateFruitTypeXMLEntries(xmlFile)
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        local isLoadFailure = false
        THUtils.call(function()
            if isUserDefaults and not rSuccess then
                THUtils.errorMsg(nil, "Failed to load default fruit types, loading vanilla configuration")
                isLoadFailure = true
            end
        end)
        if isLoadFailure then
            local newIsBaseType = FruitType.UNKNOWN == nil
            return superFunc(manager, oldXMLFile, missionInfo, oldBaseDirectory, newIsBaseType, THUtils.unpack(funcArgs))
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, isBaseType, ...))
end
function THDefaultTypes.inj_loadFoliageFromXMLFile(self, superFunc, desc, filename, ...)
    local allowFruitTypeLoad = true
    local isExistingFruitType = false
    if desc ~= nil and filename ~= nil then
        THUtils.call(function()
            if fileExists(filename) then
                local fruitXMLFile = THUtils.loadXMLFile("THFruitType", filename, nil, false)
                if THUtils.getNoNil(fruitXMLFile, 0) > 0 then
                    local fruitTypeName = getXMLString(fruitXMLFile, "foliageType.fruitType#name")
                    if fruitTypeName ~= nil then
                        local defaultTypeData = self.defaultTypes.fruitTypes
                        if defaultTypeData ~= nil then
                            if defaultTypeData:getIsBlackListItem(fruitTypeName) then
                                THUtils.displayMsg("Removing fruit type: %s", fruitTypeName)
                                allowFruitTypeLoad = false
                            end
                        end
                        if allowFruitTypeLoad then
                            local otherDesc = self.fruitTypeManager:getFruitTypeByName(fruitTypeName)
                            if otherDesc ~= nil and otherDesc.index ~= desc.index then
                                THUtils.errorMsg(false, "Fruit type %q already registered at index %s, updating existing entry...", fruitTypeName, otherDesc.index)
                                desc = otherDesc
                                isExistingFruitType = true
                            end
                        end
                    end
                    THUtils.deleteXMLFile(fruitXMLFile)
                end
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        if rSuccess then
            THUtils.call(function()
                if isExistingFruitType then
                    rSuccess = false
                end
            end)
        end
        return rSuccess, ...
    end
    if allowFruitTypeLoad then
        return appendFunc(superFunc(desc, filename, ...))
    end
    return false
end
function THDefaultTypes.inj_loadWeedSystemReplacements(self, superFunc, weedSystem, xmlFile, xmlKey, ...)
    if xmlFile ~= nil and xmlKey ~= nil then
        THUtils.call(function()
            local defaultFruitTypeData = self.defaultTypes.fruitTypes
            if defaultFruitTypeData ~= nil and next(defaultFruitTypeData.blackListItems) ~= nil then
                local replaceXMLIndex = 0
                while true do
                    local replaceXMLKey = string.format("%s.replacements(%d)", xmlKey, replaceXMLIndex)
                    if not THUtils.hasXMLProperty(xmlFile, replaceXMLKey) then
                        break
                    end
                    local fruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, replaceXMLKey, "#fruitType")
                    local isEntryRemoved = false
                    if fruitTypeName ~= nil and fruitTypeName ~= "" then
                        if defaultFruitTypeData:getIsBlackListItem(fruitTypeName) then
                            THUtils.removeXMLProperty(xmlFile, replaceXMLKey)
                            isEntryRemoved = true
                        end
                    end
                    if not isEntryRemoved then
                        replaceXMLIndex = replaceXMLIndex + 1
                    end
                end
            end
        end)
    end
    return superFunc(weedSystem, xmlFile, xmlKey, ...)
end
function THDefaultTypes.inj_inGameMenuMapLoadFilters(self, superFunc, frame, ...)
    local displayCropTypes = frame.displayCropTypes
    local fruitTypeFilter = frame.fruitTypeFilter
    local function appendFunc(...)
        if displayCropTypes ~= nil and fruitTypeFilter ~= nil then
            THUtils.call(function()
                local filterCache = {}
                for key, value in pairs(fruitTypeFilter) do
                    filterCache[key] = value
                end
                THUtils.clearTable(fruitTypeFilter)
                for cropTypeIndex, cropTypeData in pairs(displayCropTypes) do
                    if cropTypeData.fruitTypeIndex ~= nil and cropTypeData.fruitTypeIndex ~= FruitType.UNKNOWN then
                        if filterCache[cropTypeData.fruitTypeIndex] == false then
                            fruitTypeFilter[cropTypeIndex] = false
                        else
                            fruitTypeFilter[cropTypeIndex] = true
                        end
                    end
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(frame, ...))
end
function THDefaultTypes.inj_inGameMenuMapSaveFilters(self, superFunc, frame, ...)
    local displayCropTypes = frame.displayCropTypes
    local fruitTypeFilter = frame.fruitTypeFilter
    local isFruitFilterUpdated = false
    if displayCropTypes ~= nil and fruitTypeFilter ~= nil then
        THUtils.call(function()
            local otherFruitFilter = {}
            for cropTypeIndex, cropTypeData in pairs(displayCropTypes) do
                if cropTypeData.fruitTypeIndex ~= nil and cropTypeData.fruitTypeIndex ~= FruitType.UNKNOWN then
                    otherFruitFilter[cropTypeData.fruitTypeIndex] = fruitTypeFilter[cropTypeIndex] == true
                end
            end
            isFruitFilterUpdated = true
            frame.fruitTypeFilter = otherFruitFilter
        end)
    end
    local function appendFunc(...)
        if isFruitFilterUpdated then
            frame.fruitTypeFilter = fruitTypeFilter
        end
        return ...
    end
    return appendFunc(superFunc(frame, ...))
end
function THDefaultTypes.inj_inGameMenuMapGenerateOverviewOverlay(self, superFunc, frame, ...)
    local mission = g_currentMission
    local displayCropTypes = frame.displayCropTypes
    local fruitTypeFilter = frame.fruitTypeFilter
    local isFruitFilterUpdated = false
    if mission ~= nil and mission.mapOverlayGenerator ~= nil
        and displayCropTypes ~= nil and fruitTypeFilter ~= nil
    then
        THUtils.call(function()
            local missionFruitTypes = mission.mapOverlayGenerator.missionFruitTypes
            if missionFruitTypes ~= nil then
                local otherFruitFilter = {}
                for cropTypeIndex, cropTypeData in pairs(displayCropTypes) do
                    if cropTypeData.fruitTypeIndex ~= nil and cropTypeData.fruitTypeIndex ~= FruitType.UNKNOWN then
                        otherFruitFilter[cropTypeData.fruitTypeIndex] = fruitTypeFilter[cropTypeIndex] == true
                    end
                end
                for _, cropTypeData in pairs(missionFruitTypes) do
                    if cropTypeData.fruitTypeIndex ~= nil and cropTypeData.fruitTypeIndex ~= FruitType.UNKNOWN then
                        if otherFruitFilter[cropTypeData.fruitTypeIndex] == nil then
                            otherFruitFilter[cropTypeData.fruitTypeIndex] = false
                        end
                    end
                end
                isFruitFilterUpdated = true
                frame.fruitTypeFilter = otherFruitFilter
            end
        end)
    end
    local function appendFunc(...)
        if isFruitFilterUpdated then
            frame.fruitTypeFilter = fruitTypeFilter
        end
        return ...
    end
    return appendFunc(superFunc(frame, ...))
end
function THDefaultTypes.inj_loadFillTypeMapData(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, ...)
    if xmlFile ~= nil and missionInfo ~= nil then
        THUtils.call(function()
            if self.defaultTypes.fillTypes == nil then
                self:loadDefaultTypeData("fillTypes", missionInfo, xmlFile)
            end
            local defaultTypeData = self.defaultTypes.fillTypes
            if defaultTypeData ~= nil and defaultTypeData.mapXMLFile == xmlFile then
                defaultTypeData.isMapLoading = true
            end
        end)
    end
    local function appendFunc(...)
        local defaultTypeData = self.defaultTypes.fillTypes
        if defaultTypeData ~= nil then
            defaultTypeData.isMapLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, ...))
end
function THDefaultTypes.inj_loadDefaultFillTypes(self, superFunc, manager, ...)
    local defaultTypeData = self.defaultTypes.fillTypes
    if defaultTypeData ~= nil and defaultTypeData.xmlFile ~= nil then
        defaultTypeData.isDefaultLoading = true
    end
    local function appendFunc(...)
        if defaultTypeData ~= nil then
            defaultTypeData.isDefaultLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, ...))
end
function THDefaultTypes.inj_loadFillTypes(self, superFunc, manager, xmlFile, baseDirectory, isBaseType, modName, ...)
    local defaultTypeData = self.defaultTypes.fillTypes
    local isUserDefaults = false
    local oldXMLFile = xmlFile
    local oldModName = modName
    local oldBaseDirectory = baseDirectory
    local funcArgs = THUtils.pack(...)
    if defaultTypeData ~= nil then
        THUtils.call(function()
            if isBaseType == true
                and defaultTypeData.isDefaultLoading
                and defaultTypeData.xmlFile ~= nil
            then
                xmlFile = defaultTypeData.xmlFile
                modName = defaultTypeData.mapModName
                baseDirectory = defaultTypeData.mapModPath
                THUtils.displayMsg("Loading default fill types from: %s", defaultTypeData.xmlFilename)
                isUserDefaults = true
            end
            if xmlFile ~= nil and next(defaultTypeData.blackListItems) ~= nil then
                self:updateFillTypeXMLEntries(xmlFile)
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        local isLoadFailure = false
        THUtils.call(function()
            if isUserDefaults and not rSuccess then
                THUtils.errorMsg(nil, "Failed to load default fill types, loading vanilla configuration")
                isLoadFailure = true
            end
        end)
        if isLoadFailure then
            local newIsBaseType = FillType.UNKNOWN == nil
            return superFunc(manager, oldXMLFile, oldBaseDirectory, newIsBaseType, oldModName, THUtils.unpack(funcArgs))
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(manager, xmlFile, baseDirectory, isBaseType, modName, ...))
end
function THDefaultTypes.inj_addFillType(self, superFunc, manager, desc, ...)
    local allowFillTypeAdd = true
    if type(desc) == THValueType.TABLE and desc.name ~= nil then
        THUtils.call(function()
            local defaultTypeData = self.defaultTypes.fillTypes
            if defaultTypeData ~= nil then
                local descId = tostring(desc.name):upper()
                if defaultTypeData:getIsBlackListItem(descId) then
                    THUtils.displayMsg("Removing fill type: %s", descId)
                    allowFillTypeAdd = false
                end
            end
            if allowFillTypeAdd then
                local otherDesc = manager:getFillTypeByName(desc.name)
                if otherDesc ~= nil then
                    THUtils.errorMsg(false, "Fill type %q already registered at index %s, skipping addition...", otherDesc.name, otherDesc.index)
                    allowFillTypeAdd = false
                end
            end
        end)
    end
    if not allowFillTypeAdd then
        return false
    end
    return superFunc(manager, desc, ...)
end
function THDefaultTypes.inj_loadBaleDataFromXML(self, superFunc, manager, bale, xmlFile, baseDirectory, ...)
    if bale ~= nil and xmlFile ~= nil then
        THUtils.call(function()
            local defaultFillTypeData = self.defaultTypes.fillTypes
            local fillTypesKey = "bale.fillTypes"
            if defaultFillTypeData ~= nil and next(defaultFillTypeData.blackListItems) ~= nil
                and THUtils.hasXMLProperty(xmlFile, fillTypesKey)
            then
                local isBaleFilenameShown = false
                local entryXMLIndex = 0
                while true do
                    local entryKey = string.format("%s.fillType(%d)", fillTypesKey, entryXMLIndex)
                    if not THUtils.hasXMLProperty(xmlFile, entryKey) then
                        break
                    end
                    local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, entryKey, "#name")
                    if fillTypeName ~= nil and fillTypeName ~= ""
                        and defaultFillTypeData:getIsBlackListItem(fillTypeName)
                    then
                        THUtils.removeXMLProperty(xmlFile, entryKey)
                        if THDebugUtil.getIsEnabled(debugFlagId) then
                            if not isBaleFilenameShown then
                                THUtils.displayMsg("Bale: %s", bale.xmlFilename)
                                isBaleFilenameShown = true
                            end
                            THUtils.displayMsg("- Removing fill type: %s", fillTypeName)
                        end
                    else
                        entryXMLIndex = entryXMLIndex + 1
                    end
                end
            end
        end)
    end
    return superFunc(manager, bale, xmlFile, baseDirectory, ...)
end
function THDefaultTypes.inj_loadHeightTypeMapData(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, ...)
    if xmlFile ~= nil and missionInfo ~= nil then
        THUtils.call(function()
            if self.defaultTypes.densityMapHeightTypes == nil then
                self:loadDefaultTypeData("densityMapHeightTypes", missionInfo, xmlFile)
            end
            local defaultTypeData = self.defaultTypes.densityMapHeightTypes
            if defaultTypeData ~= nil and defaultTypeData.mapXMLFile == xmlFile then
                defaultTypeData.isMapLoading = true
            end
        end)
    end
    local function appendFunc(...)
        local defaultTypeData = self.defaultTypes.densityMapHeightTypes
        if defaultTypeData ~= nil then
            defaultTypeData.isMapLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, ...))
end
function THDefaultTypes.inj_loadDefaultHeightTypes(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, ...)
    local defaultTypeData = self.defaultTypes.densityMapHeightTypes
    if defaultTypeData ~= nil and defaultTypeData.xmlFile ~= nil then
        defaultTypeData.isDefaultLoading = true
    end
    local function appendFunc(...)
        if defaultTypeData ~= nil then
            defaultTypeData.isDefaultLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, ...))
end
function THDefaultTypes.inj_loadHeightTypes(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, isBaseType, ...)
    local defaultTypeData = self.defaultTypes.densityMapHeightTypes
    local isUserDefaults = false
    local oldXMLFile = xmlFile
    local oldBaseDirectory = baseDirectory
    local funcArgs = THUtils.pack(...)
    if defaultTypeData ~= nil then
        THUtils.call(function()
            if isBaseType == true
                and defaultTypeData.isDefaultLoading
                and defaultTypeData.xmlFile ~= nil
            then
                xmlFile = defaultTypeData.xmlFile
                baseDirectory = defaultTypeData.mapModPath
                THUtils.displayMsg("Loading default height types from: %s", defaultTypeData.xmlFilename)
                isUserDefaults = true
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        local isLoadFailure = false
        THUtils.call(function()
            if isUserDefaults and not rSuccess then
                THUtils.errorMsg(nil, "Failed to load default height types, loading vanilla configuration")
                isLoadFailure = true
            end
        end)
        if isLoadFailure then
            local newIsBaseType = false
            return superFunc(manager, oldXMLFile, missionInfo, oldBaseDirectory, newIsBaseType, THUtils.unpack(funcArgs))
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, isBaseType, ...))
end
function THDefaultTypes.inj_loadHeightTypeFromXML(self, superFunc, manager, xmlFile, xmlKey, isBaseType, ...)
    local defaultTypeData = self.defaultTypes.densityMapHeightTypes
    local allowHeightTypeAdd = true
    if defaultTypeData ~= nil
        and xmlFile ~= nil and xmlKey ~= nil
    then
        THUtils.call(function()
            local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, xmlKey, "#fillTypeName")
            if fillTypeName ~= nil and fillTypeName ~= "" then
                local defaultFillTypeData = self.defaultTypes.fillTypes
                if defaultTypeData:getIsBlackListItem(fillTypeName)
                    or (defaultFillTypeData ~= nil and defaultFillTypeData:getIsBlackListItem(fillTypeName))
                then
                    allowHeightTypeAdd = false
                    THUtils.displayMsg("Removing height type: %s", fillTypeName)
                end
            end
        end)
    end
    if allowHeightTypeAdd then
        return superFunc(manager, xmlFile, xmlKey, isBaseType, ...)
    end
end
function THDefaultTypes.inj_loadStoreItemMapData(self, superFunc, manager, xmlFile, missionInfo, baseDirectory, ...)
    if xmlFile ~= nil and missionInfo ~= nil then
        THUtils.call(function()
            if self.defaultTypes.storeItems == nil then
                self:loadDefaultTypeData("storeItems", missionInfo, xmlFile)
            end
            local defaultTypeData = self.defaultTypes.storeItems
            if defaultTypeData ~= nil and defaultTypeData.mapXMLFile == xmlFile then
                defaultTypeData.isMapLoading = true
            end
        end)
    end
    local function appendFunc(...)
        local defaultTypeData = self.defaultTypes.storeItems
        if defaultTypeData ~= nil then
            defaultTypeData.isMapLoading = false
        end
        return ...
    end
    return appendFunc(superFunc(manager, xmlFile, missionInfo, baseDirectory, ...))
end
function THDefaultTypes.inj_loadStoreItemsFromXML(self, superFunc, manager, xmlFilename, baseDirectory, customEnv, ...)
    local defaultTypeData = self.defaultTypes.storeItems
    local isUserDefaults = false
    local oldXMLFilename = xmlFilename
    local oldCustomEnv = customEnv
    local oldBaseDirectory = baseDirectory
    if xmlFilename ~= nil then
        THUtils.call(function()
            if defaultTypeData ~= nil and defaultTypeData.xmlFile ~= nil then
                local defaultXMLFilename = manager:getDefaultStoreItemsFilename()
                if defaultXMLFilename == xmlFilename
                    and (baseDirectory == nil or baseDirectory == "")
                    and (customEnv == nil or customEnv == "")
                then
                    xmlFilename = defaultTypeData.xmlFilename
                    baseDirectory = defaultTypeData.mapModPath
                    customEnv = defaultTypeData.mapModName
                    defaultTypeData.isDefaultLoading = true
                    isUserDefaults = true
                end
            end
        end)
    end
    local function appendFunc(...)
        if isUserDefaults and defaultTypeData ~= nil then
            THUtils.call(function()
                self.taskManager:addSubtask(function()
                    defaultTypeData.isDefaultLoading = false
                end)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(manager, xmlFilename, baseDirectory, customEnv, ...))
end
function THDefaultTypes.inj_loadStoreItem(self, superFunc, manager, xmlFilename, baseDirectory, customEnv, isMod, isBundleItem, dlcTitle, extraContentId, ignoreAdd, ...)
    local isLoadAllowed = true
    if xmlFilename ~= nil then
        THUtils.call(function()
            local defaultTypeData = self.defaultTypes.storeItems
            if defaultTypeData ~= nil then
                if defaultTypeData:getIsBlackListItem(xmlFilename, customEnv) then
                    THUtils.displayMsg("Removing store item: %s", xmlFilename)
                    isLoadAllowed = false
                end
            end
        end)
    end
    if isLoadAllowed then
        return superFunc(manager, xmlFilename, baseDirectory, customEnv, isMod, isBundleItem, dlcTitle, extraContentId, ignoreAdd, ...)
    end
end
function THDefaultTypes.inj_loadVehicleSpecValueCombinations(self, superFunc, xmlFile, customEnv, baseDirectory, ...)
    if xmlFile ~= nil then
        THUtils.call(function()
            local defaultTypeData = self.defaultTypes.storeItems
            if defaultTypeData ~= nil and next(defaultTypeData.blackListItems) ~= nil then
                local combinationIndex = 0
                while true do
                    local combinationKey = string.format("vehicle.storeData.specs.combination(%d)", combinationIndex)
                    if not THUtils.hasXMLProperty(xmlFile, combinationKey) then
                        break
                    end
                    local xmlFilename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, combinationKey, "#xmlFilename")
                    local isRemoved = false
                    if xmlFilename ~= nil and xmlFilename ~= "" then
                        if defaultTypeData:getIsBlackListItem(xmlFilename, customEnv) then
                            THUtils.removeXMLProperty(xmlFile, combinationKey)
                            isRemoved = true
                        end
                    end
                    if not isRemoved then
                        combinationIndex = combinationIndex + 1
                    end
                end
            end
        end)
    end
    return superFunc(xmlFile, customEnv, baseDirectory, ...)
end
THUtils.call(function()
    local self = THDefaultTypes.new()
    if self ~= nil then
        _G.g_thDefaultTypes = self
        THUtils.setFunctionHook("FruitTypeManager", "loadMapData", false, false, self, THDefaultTypes.inj_loadFruitTypeMapData)
        THUtils.setFunctionHook("FruitTypeManager", "loadDefaultTypes", false, false, self, THDefaultTypes.inj_loadDefaultFruitTypes)
        THUtils.setFunctionHook("FruitTypeManager", "loadFruitTypes", false, false, self, THDefaultTypes.inj_loadFruitTypes)
        THUtils.setFunctionHook("FruitTypeDesc", "loadFromFoliageXMLFile", false, false, self, THDefaultTypes.inj_loadFoliageFromXMLFile)
        THUtils.setFunctionHook("InGameMenuMapFrame", "loadFilters", false, false, self, THDefaultTypes.inj_inGameMenuMapLoadFilters)
        THUtils.setFunctionHook("InGameMenuMapFrame", "saveFilters", false, false, self, THDefaultTypes.inj_inGameMenuMapSaveFilters)
        THUtils.setFunctionHook("InGameMenuMapFrame", "generateOverviewOverlay", false, false, self, THDefaultTypes.inj_inGameMenuMapGenerateOverviewOverlay)
        THUtils.setFunctionHook("WeedSystem", "loadReplacements", false, false, self, THDefaultTypes.inj_loadWeedSystemReplacements)
        THUtils.setFunctionHook("FillTypeManager", "loadMapData", false, false, self, THDefaultTypes.inj_loadFillTypeMapData)
        THUtils.setFunctionHook("FillTypeManager", "loadDefaultTypes", false, false, self, THDefaultTypes.inj_loadDefaultFillTypes)
        THUtils.setFunctionHook("FillTypeManager", "loadFillTypes", false, false, self, THDefaultTypes.inj_loadFillTypes)
        THUtils.setFunctionHook("FillTypeManager", "addFillType", false, false, self, THDefaultTypes.inj_addFillType)
        THUtils.setFunctionHook("BaleManager", "loadBaleDataFromXML", false, false, self, THDefaultTypes.inj_loadBaleDataFromXML)
        THUtils.setFunctionHook("DensityMapHeightManager", "loadMapData", false, false, self, THDefaultTypes.inj_loadHeightTypeMapData)
        THUtils.setFunctionHook("DensityMapHeightManager", "loadDefaultTypes", false, false, self, THDefaultTypes.inj_loadDefaultHeightTypes)
        THUtils.setFunctionHook("DensityMapHeightManager", "loadDensityMapHeightTypes", false, false, self, THDefaultTypes.inj_loadHeightTypes)
        THUtils.setFunctionHook("DensityMapHeightManager", "loadDensityMapHeightTypeFromXML", false, false, self, THDefaultTypes.inj_loadHeightTypeFromXML)
        THUtils.setFunctionHook("StoreManager", "loadMapData", false, false, self, THDefaultTypes.inj_loadStoreItemMapData)
        THUtils.setFunctionHook("StoreManager", "loadItemsFromXML", false, false, self, THDefaultTypes.inj_loadStoreItemsFromXML)
        THUtils.setFunctionHook("StoreManager", "loadItem", false, false, self, THDefaultTypes.inj_loadStoreItem)
        THUtils.setFunctionHook("Vehicle", "loadSpecValueCombinations", false, false, self, THDefaultTypes.inj_loadVehicleSpecValueCombinations)
        THDebugUtil.createConsoleCommands(self)
    end
end)