-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

THSoilMap = {}
local THSoilMap_mt = Class(THSoilMap)
THSoilMap.SOIL_MAP_ID = {
    MAPUS = "mapUS",
    MAPAS = "mapAS",
    MAPEU = "mapEU",
    MAPFR = "mapFR",
    MAPSA = "mapSA",
    MAP_ALPINE = "mapAlpine",
    MAP_GENERIC = "generic"
}
THUtils.createEnumTable(THSoilMap.SOIL_MAP_ID)
THSoilMap.BIT_VECTOR_NAME = {
    SOIL_MAP = "SoilMap",
    SEED_RATE_MAP = "SeedRateMap",
    COVER_MAP = "coverMap",
    NITROGEN_MAP = "nitrogenMap",
    NITROGEN_NOISE_MAP = "nitrogenNoiseMap",
    PH_MAP = "phMap",
    PH_NOISE_MAP = "pHNoiseMap",
    YIELD_MAP = "YieldMap",
}
THUtils.createEnumTable(THSoilMap.BIT_VECTOR_NAME)
local function initScript()
    local self = THSoilMap.new()
    if self ~= nil then
        _G.g_thSoilMap = self
        THUtils.setFunctionHook(g_thGlobalEnv, "loadBitVectorMapNew", false, false, self, THSoilMap.inj_loadBitVectorMapNew)
        THUtils.setFunctionHook(g_thGlobalEnv, "loadBitVectorMapFromFile", false, false, self, THSoilMap.inj_loadBitVectorMapFromFile)
        THUtils.setFunctionHook(g_thGlobalEnv, "getBitVectorMapSize", false, false, self, THSoilMap.inj_getBitVectorMapSize)
        THUtils.setFunctionHook(THPFConfig, "loadFromMapXML", false, false, self, THSoilMap.inj_loadFromMapXML)
        THUtils.setFunctionHook(THPFConfig, "onLoadMapFinished", false, false, self, THSoilMap.inj_onLoadMapFinished)
    end
end
function THSoilMap.new(customMt)
    customMt = customMt or THSoilMap_mt
    local self = setmetatable({}, customMt)
    self.modulePath = g_thPFConfig.coreData.mod.path .. "modules/soilMap/"
    self.soilMapSize = 1024
    self.soilMapPath = self.modulePath .. "soilMaps/"
    self.noiseMapPath = self.modulePath .. "noiseMaps/"
    self.bypassSoilMapSizeLimit = false
    self.pauseBitVectorProcessing = false
    return self
end
function THSoilMap.getBitVectorMapSize(self, bitVectorMap)
    self.pauseBitVectorProcessing = true
    local sizeX, sizeY = getBitVectorMapSize(bitVectorMap)
    self.pauseBitVectorProcessing = false
    return sizeX, sizeY
end
function THSoilMap.setBitVectorEnvironmentValues(_, bitVectorMap, sizeX, sizeY, filename)
    local pfData = g_thPFConfig.precisionFarming
    if THUtils.argIsValid(type(bitVectorMap) == THValueType.NUMBER and bitVectorMap > 0, "bitVectorMap", bitVectorMap)
        and THUtils.argIsValid(type(sizeX) == THValueType.NUMBER and sizeX > 0, "sizeX", sizeX)
        and THUtils.argIsValid(type(sizeY) == THValueType.NUMBER and sizeY > 0, "sizeY", sizeY)
        and THUtils.argIsValid(filename == nil or type(filename) == THValueType.STRING, "filename", filename)
        and pfData.isFound
    then
        local pfTarget = pfData.target
        local bitVectorName = getName(bitVectorMap)
        if bitVectorName == THSoilMap.BIT_VECTOR_NAME.SOIL_MAP then
            if filename ~= nil then
                pfTarget.soilMap.loadFilename = filename
            end
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.SEED_RATE_MAP then
            pfTarget.seedRateMap.sizeX = sizeX
            pfTarget.seedRateMap.sizeY = sizeY
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.COVER_MAP then
            pfTarget.coverMap.sizeX = sizeX
            pfTarget.coverMap.sizeY = sizeY
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.NITROGEN_MAP then
            pfTarget.nitrogenMap.sizeX = sizeX
            pfTarget.nitrogenMap.sizeY = sizeY
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.NITROGEN_NOISE_MAP then
            pfTarget.nitrogenMap.noiseResolution = sizeX
            if filename ~= nil then
                pfTarget.nitrogenMap.noiseFilename = filename
            end
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.PH_MAP then
            pfTarget.pHMap.sizeX = sizeX
            pfTarget.pHMap.sizeY = sizeY
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.PH_NOISE_MAP then
            pfTarget.pHMap.noiseResolution = sizeX
            if filename ~= nil then
                pfTarget.pHMap.noiseFilename = filename
            end
        elseif bitVectorName == THSoilMap.BIT_VECTOR_NAME.YIELD_MAP then
            pfTarget.yieldMap.sizeX = sizeX
            pfTarget.yieldMap.sizeY = sizeY
        end
        return true
    end
    return false
end
function THSoilMap.inj_loadBitVectorMapNew(self, superFunc, bitVectorMap, sizeX, sizeY, ...)
    local mission = g_thPFConfig.coreData.mission
    local pfData = g_thPFConfig.precisionFarming
    local bitVectorName = nil
    local prependSuccess = false
    if mission ~= nil and pfData.isFound
        and type(bitVectorMap) == THValueType.NUMBER and bitVectorMap > 0
        and type(sizeX) == THValueType.NUMBER and sizeX <= 1024
        and type(sizeY) == THValueType.NUMBER and sizeY <= 1024
    then
        THUtils.pcall(function()
            bitVectorName = getName(bitVectorMap)
            if bitVectorName ~= nil then
                if THSoilMap.BIT_VECTOR_NAME:getId(bitVectorName) ~= nil then
                    if bitVectorName == THSoilMap.BIT_VECTOR_NAME.SOIL_MAP then
                        if self.soilMapName ~= nil then
                            sizeX = 2048
                            sizeY = 2048
                        end
                    elseif self.soilMapSize > 1024 then
                        sizeX = self.soilMapSize
                        sizeY = self.soilMapSize
                    end
                    prependSuccess = true
                else
                    if THUtils.getIsDebugEnabled() then
                        THUtils.displayMsg("Creating new <= 1024 bitVectorMap: %s", bitVectorName)
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        if prependSuccess then
            THUtils.pcall(function()
                local bitVectorSizeX, bitVectorSizeY = self:getBitVectorMapSize(bitVectorMap)
                if bitVectorSizeX ~= nil and bitVectorSizeX > 1024
                    and bitVectorSizeY ~= nil and bitVectorSizeY > 1024
                then
                    if bitVectorName == THSoilMap.BIT_VECTOR_NAME.SOIL_MAP then
                        self.soilMapSize = bitVectorSizeX
                    end
                    self:setBitVectorEnvironmentValues(bitVectorMap, bitVectorSizeX, bitVectorSizeY)
                    THUtils.displayMsg("Injecting bitVectorMap: %s (%d x %d)", bitVectorName, bitVectorSizeX, bitVectorSizeY)
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(bitVectorMap, sizeX, sizeY, ...))
end
function THSoilMap.inj_loadBitVectorMapFromFile(self, superFunc, bitVectorMap, filename, numChannels, ...)
    local pfData = g_thPFConfig.precisionFarming
    local bitVectorName = nil
    local prependSuccess = false
    if pfData.isFound
        and type(bitVectorMap) == THValueType.NUMBER and bitVectorMap > 0
        and type(filename) == THValueType.STRING and filename ~= ""
    then
        THUtils.pcall(function()
            local PrecisionFarming = pfData.mod.environment.PrecisionFarming
            bitVectorName = getName(bitVectorMap)
            if bitVectorName ~= nil and THSoilMap.BIT_VECTOR_NAME:getId(bitVectorName) ~= nil then
                local isDefaultFilename = string.find(filename, PrecisionFarming.BASE_DIRECTORY, nil, true) ~= nil
                if bitVectorName == THSoilMap.BIT_VECTOR_NAME.SOIL_MAP then
                    if isDefaultFilename and self.soilMapFilename ~= nil then
                        filename = self.soilMapFilename
                    end
                elseif self.soilMapSize > 1024 then
                    if bitVectorName == THSoilMap.BIT_VECTOR_NAME.NITROGEN_NOISE_MAP
                        or bitVectorName == THSoilMap.BIT_VECTOR_NAME.PH_NOISE_MAP
                    then
                        if isDefaultFilename then
                            local newFilename = string.format("ph_noise_%s.png", self.soilMapSize)
                            local absNewFilename = THUtils.getFilename(newFilename, self.noiseMapPath)
                            if THUtils.getFileExists(absNewFilename) then
                                filename = absNewFilename
                            end
                        end
                    end
                end
                prependSuccess = true
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        if rSuccess and prependSuccess then
            THUtils.pcall(function()
                local bitVectorSizeX, bitVectorSizeY = self:getBitVectorMapSize(bitVectorMap)
                if bitVectorSizeX ~= nil and bitVectorSizeX > 1024
                    and bitVectorSizeY ~= nil and bitVectorSizeY > 1024
                then
                    if bitVectorName == THSoilMap.BIT_VECTOR_NAME.SOIL_MAP then
                        self.soilMapSize = bitVectorSizeX
                        self.bypassSoilMapSizeLimit = true
                    end
                    self:setBitVectorEnvironmentValues(bitVectorMap, bitVectorSizeX, bitVectorSizeY, filename)
                    THUtils.displayMsg("Loading bitVectorMap %s: (%d x %d) >> %s", bitVectorName, bitVectorSizeX, bitVectorSizeY, filename)
                end
            end)
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(bitVectorMap, filename, numChannels, ...))
end
function THSoilMap.inj_getBitVectorMapSize(self, superFunc, bitVectorMap, ...)
    if self.pauseBitVectorProcessing then
        return superFunc(bitVectorMap, ...)
    end
    local pfData = g_thPFConfig.precisionFarming
    local function appendFunc(rSizeX, rSizeY, ...)
        if pfData.isFound
            and type(bitVectorMap) == THValueType.NUMBER and bitVectorMap > 0
        then
            THUtils.pcall(function()
                local bitVectorName = getName(bitVectorMap)
                if bitVectorName == THSoilMap.BIT_VECTOR_NAME.SOIL_MAP then
                    if self.bypassSoilMapSizeLimit
                        and (rSizeX > 1024 or rSizeY > 1024)
                    then
                        THUtils.displayMsg("Bypassing %s limit of 1024 x 1024", bitVectorName)
                        THUtils.displayMsg("- new size: %d x %d", rSizeX, rSizeY)
                        rSizeX = 1024
                        rSizeY = 1024
                        self.bypassSoilMapSizeLimit = false
                    end
                end
            end)
        end
        return rSizeX, rSizeY, ...
    end
    return appendFunc(superFunc(bitVectorMap, ...))
end
function THSoilMap.inj_loadFromMapXML(self, superFunc, parent, xmlFile, xmlKey, ...)
    local function appendFunc(rSuccess, ...)
        if rSuccess then
            THUtils.pcall(function()
                local soilMapKey = xmlKey .. ".soilMap"
                local soilMapId = xmlFile:getString(soilMapKey .. "#id")
                local soilMapSize = xmlFile:getInt(soilMapKey .. "#size", 2048)
                if soilMapId ~= nil and soilMapId ~= "" then
                    local soilMapName = THSoilMap.SOIL_MAP_ID:getValue(soilMapId)
                    if soilMapName == nil then
                        local soilMapIds = THSoilMap.SOIL_MAP_ID:getValues()
                        THUtils.xmlErrorMsg(xmlKey, nil, THMessage.INVALID_VALUE, "soilMapId", soilMapId)
                        if soilMapIds ~= nil then
                            THUtils.displayMsg("- available values:")
                            for _, value in pairs(soilMapIds) do
                                THUtils.displayMsg("  + %s", value)
                            end
                        end
                    else
                        local soilMapFilename = string.format("%s%s/%s.png", self.soilMapPath, soilMapSize, soilMapName)
                        if not THUtils.getFileExists(soilMapFilename) then
                            THUtils.xmlErrorMsg(soilMapKey, nil, THMessage.FILE_NOT_FOUND, soilMapFilename)
                        else
                            self.soilMapName = soilMapName
                            self.soilMapFilename = soilMapFilename
                        end
                    end
                end
            end)
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(parent, xmlFile, xmlKey, ...))
end
function THSoilMap.inj_onLoadMapFinished(self, superFunc, parent, mission, ...)
    local pfData = g_thPFConfig.precisionFarming
    local function appendFunc(...)
        THUtils.pcall(function()
            if THUtils.getIsDebugEnabled() then
                local pfTarget = pfData.target
                THUtils.displayMsg("THSoilMap information:")
                THUtils.printTable(self)
                if self.soilMapSize > 1024 then
                    THUtils.displayMsg("SoilMap information:")
                    THUtils.printTableValues(pfTarget.soilMap)
                    THUtils.displayMsg("SeedRateMap information:")
                    THUtils.printTableValues(pfTarget.seedRateMap)
                    THUtils.displayMsg("CoverMap information:")
                    THUtils.printTableValues(pfTarget.coverMap)
                    THUtils.displayMsg("NitrogenMap information:")
                    THUtils.printTableValues(pfTarget.nitrogenMap)
                    THUtils.displayMsg("PHMap information:")
                    THUtils.printTableValues(pfTarget.pHMap)
                    THUtils.displayMsg("YieldMap information:")
                    THUtils.printTableValues(pfTarget.yieldMap)
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(parent, mission, ...))
end
THUtils.pcall(initScript)